//
// This is the base class for all events that you want to create. It loads in the most basic parameters (like Delay and Duration) that are used in every event.
// 
/******************************************************************************/
class BaseEvent : Game.Obj
{
public:
            BaseEvent();
   virtual ~BaseEvent();
   
   // Override the create() in each inherited class to load your own parameters. If you do, don't forget to call super.create();
   virtual void create(Object &obj);
   // Override the AfterCreate in each inherited class to link the correct events to be called or objects to be referenced. 
   virtual void AfterCreate(              ) { }

   // Override the update() only to add your own logic that needs updating.      If you do, don't forget to call super.update();
   virtual Bool update();
   virtual UInt drawPrepare() { return 0 ;}
   
   virtual Vec    pos   (                ) final { return 0; } // get position
   virtual void   pos   (C Vec    &pos   ) final {           } // set position
   virtual Matrix matrix(                ) final { return 0; } // get matrix
   virtual void   matrix(C Matrix &matrix) final {           } // set matrix, for most classes 'matrix' should be normalized 
   
   void Triggered();
   
protected:

   bool m_EventCalled, m_LogOutput, m_LogDetailed;
   Str m_EditorInfo;
   
private:

   // Override these to define what the event does
   virtual void CallTriggeredAction() = NULL;
   virtual void StopTriggeredAction() = NULL;

   flt m_Duration, m_Timer, m_Delay;
   bool m_StartTimer, m_Active, m_FirstTick;
};
/******************************************************************************/
BaseEvent::BaseEvent() : m_Duration(0), m_StartTimer(false), m_Timer(0), m_Delay(0), m_Active(true), m_EventCalled(false), m_FirstTick(true), m_LogOutput(false), m_LogDetailed(false)            
{
   
}
BaseEvent::~BaseEvent()
{
   
}
/******************************************************************************/
void BaseEvent::create(Object &obj)
{
   // Getting the parameters from the object in the world
   if(Param *p = obj.findParam("-Debug_LogOutput"))   m_LogOutput   = p.asBool(); 
   if(Param *p = obj.findParam("-Debug_DetailedLog")) m_LogDetailed = p.asBool(); 
   if(Param *p = obj.findParam("-EDITOR-INFO"))       m_EditorInfo  = p.asText();
   if(Param *p = obj.findParam("Active"))             m_Active      = p.asBool(); // Active is set to true by default, so you don't *have* to add the param in the object class
   if(Param *p = obj.findParam("Delay"))              m_Delay       = p.asFlt();
   if(Param *p = obj.findParam("Duration"))           m_Duration    = p.asFlt();  // -1 = infinite 
   
}
/******************************************************************************/
bool BaseEvent::update()
{
   // On the first frame, call AfterCreate(). This method allows you to initialize all events/objects that need to be called/referenced
   if(m_FirstTick)
   {
      AfterCreate();
      
      m_FirstTick = false;
   }
   
   // Event has been triggered
   if(m_StartTimer)
   {      
      // Fire the action (after appropriate delay-time)
      if(m_Timer >= m_Delay && !m_EventCalled)
      {
         CallTriggeredAction();
         m_EventCalled = true;  
      }
      
      // A duration of -1 means this event never stops
      if(m_Duration != -1)
      {
         // Stop the event
         if(m_Timer >= (m_Duration + m_Delay))
         {
            // Reset the timer and the eventcalled, in case this event can get called again.
            m_StartTimer = false;
            m_EventCalled = false;
            
            // Call whatever needs to happen when the event ends
            StopTriggeredAction();
         } 
      } 
      
      m_Timer += Time.d(); 
   }
   
   return true;
}
/******************************************************************************/
void BaseEvent::Triggered()
{
   // Start the event, if it's set to "Active"
   if(m_Active)
   {
      m_EventCalled = false;
      m_StartTimer = true; 
      m_Timer = 0;      
   }
}
/******************************************************************************/